/**
* \file: AudioChannel.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: CarPlay
*
* \author: J. Harder / ADIT/SW1 / jharder@de.adit-jv.com
*
* \copyright (c) 2013-2014 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <TickUtils.h>
#include <AudioUtils.h>
#include "AudioChannel.h"
#include "utils/Statistics.h"
#include "utils/Utils.h"
#include "Session.h"

#include <inttypes.h>

namespace adit { namespace carplay
{

using namespace std;

AudioChannel::AudioChannel()
{
    session = nullptr;
    config = nullptr;
    hasInput = false;
    verbose = false;
    inputContext = nullptr;
    inputCallback = nullptr;
    outputContext = nullptr;
    outputCallback = nullptr;
    latencyMs = 0;
    memset(&format, 0, sizeof(format));
    channel = AudioChannelType_Main;
    hostStartTime = 0;
    hostInStartTime = 0;
    sampleNumber = 0;

    previousHostTime = 0;
    previousSampleTime = 0;

    previousWriteHostTime = 0;
    previousWriteSampleTime = 0;

    firstOutSample = true;
    firstInSample = true;
}

AudioChannel::~AudioChannel()
{
    LOGD_DEBUG((dipo, "AudioChannel stopped"));
}

bool AudioChannel::Initialize(IConfiguration& inConfig)
{
    config = &inConfig;

    auto ptr = Factory::Instance()->Create(inConfig.GetItem("IAudioOutAdapter", ""));
    out = move(unique_ptr<IAudioOutAdapter>(static_cast<IAudioOutAdapter*>(ptr)));
    if (out != nullptr && !out->Initialize(inConfig, *this))
    {
        // in case of IAudioOutAdapter errors continue with disabled audio out
        LOG_ERROR((dipo, "could not initialize IAudioOutAdapter! Audio out id disabled!"));
        out = nullptr; // delete object
        return false;
    }

    // create input adapter with Prepare message as we do not know yet if we have input

    latencyMs = inConfig.GetNumber("audio-out-system-latency-ms", 20);

    string verboseConfig;
    if (config->TryGetItem("enable-verbose-logging", verboseConfig))
    {
        verbose = 0 != config->GetNumber("enable-verbose-logging", 0);
        if (verbose)
        {
            LOG_WARN((dipo, "audio: verbose logging enabled, do not use in production!"));
        }
    }

    return true;
}

bool AudioChannel::Prepare(AudioChannelType inChannelType, bool inHasInput,
        AudioFormatStruct inFormat, const string& inAudioType)
{
    if (config == nullptr || session == nullptr)
    {
        LOG_ERROR((dipo, "AudioChannel is not initialized"));
        return false;
    }

    LOGD_DEBUG((dipo, "audio stream start (is main: %d, has input: %d, sample rate: %d, vocoderSampleRate: %f " \
            "bits per channel: %d, channels: %d, audioType: %s)",
            inChannelType == AudioChannelType_Main, inHasInput, inFormat.SampleRate, inFormat.vocoderSampleRate,
            inFormat.BitsPerChannel, inFormat.Channels,
            inAudioType.c_str()));
    hasInput = inHasInput;
    format = inFormat;
    channel = inChannelType;

    if (hasInput)
    {
        auto ptr = Factory::Instance()->Create(config->GetItem("IAudioInAdapter", ""));
        in = move(unique_ptr<IAudioInAdapter>(static_cast<IAudioInAdapter*>(ptr)));
        if (in != nullptr && !in->Initialize(*config, *this))
        {
            LOG_ERROR((dipo, "could not initialize IAudioInAdapter! Audio in is disabled!"));
            in = nullptr; // delete object
            return false;
        }
    }

    // signal also control adapter about the possible audio type change
    session->AudioPrepare(channel, inAudioType);

    if (out != nullptr && !out->Prepare(inFormat, channel, inAudioType))
    {
        LOG_ERROR((dipo, "could not prepare IAudioOutAdapter"));
        return false;
    }

    if (inHasInput && in != nullptr && !in->Prepare(inFormat, inAudioType))
    {
        LOG_ERROR((dipo, "could not prepare IAudioInAdapter"));
        return false;
    }

    firstInSample = true;
    firstOutSample = true;

    return true;
}

bool AudioChannel::Start()
{
    if (out != nullptr)
    {
        if (!out->Start())
        {
            LOG_ERROR((dipo, "could not start IAudioOutAdapter"));
            return false;
        }
    }
    else
    {
        LOG_ERROR((dipo, "audio out is disabled, ignore start"));
    }

    if (hasInput && in != nullptr)
    {
        if (!in->Start())
        {
            LOG_ERROR((dipo, "could not start IAudioInAdapter"));
            return false;
        }
    }
    else if (hasInput)
    {
        LOG_ERROR((dipo, "audio in is disabled, ignore start"));
    }

    auto avChannel = channel == AudioChannelType_Main ? AVChannel_MainAudio : AVChannel_AltAudio;
    Statistics::Instance().SetAVChannel(session, avChannel, true);
    return true;
}

bool AudioChannel::Stop()
{
    // be silent in regards to disabled audio

    if (out != nullptr)
        out->Stop();
    if (in != nullptr)
        in->Stop();

    // signal also control adapter
    session->AudioStop(channel);

    auto avChannel = channel == AudioChannelType_Main ? AVChannel_MainAudio : AVChannel_AltAudio;
    Statistics::Instance().SetAVChannel(session, avChannel, false);
    return true;
}

void AudioChannel::Flush()
{
    if (out != nullptr)
	    out->Flush();
}
void AudioChannel::Read(const Samples& inSamples)
{
    const uint64_t NANOSECOND = 1000000000;

    if (outputCallback == nullptr)
        return;

    // Time stamps are used to feedback rate estimations to the iPhone.
    // They do not directly used to select the buffers that are read.

    uint64_t hostTime = GetCurrentTimeNano();
    uint32_t sampleTime = inSamples.TimeStamp;

    if (firstOutSample == true)
    {
        previousHostTime = hostTime;
        previousSampleTime = inSamples.TimeStamp;
        firstOutSample = false;
    }

    if (verbose)
    {
        uint64_t hostDiff = hostTime - previousHostTime;
        previousHostTime = hostTime;
        uint64_t sampleDiff = ((sampleTime - previousSampleTime) * NANOSECOND) / format.SampleRate;
        previousSampleTime = sampleTime;
        (void)hostDiff;
        (void)sampleDiff;

        LOGD_VERBOSE((dipo, "%s audio out: host %" PRIu64 ", sample %u, host diff %" PRIu64 ", sample diff %" PRIu64,
                (channel == AudioChannelType_Main ? "main" : "alt"), hostTime, sampleTime,
                hostDiff, sampleDiff));
    }

    // [AirPlay] Re-buffering occurs when we are reading samples faster than the iPhone can provide.
    // This can also happen when the playing queue is too large.
    outputCallback(sampleTime, hostTime, inSamples.DataPtr, inSamples.Length,
            outputContext);
}

void AudioChannel::Write(const Samples& inSamples)
{
    const uint64_t NANOSECOND = 1000000000;

    if (inputCallback == nullptr)
        return;

    uint64_t hosttime = GetCurrentTimeNano();

    if (firstInSample == true)
    {
        hostInStartTime = hosttime;
        sampleNumber = 0;
        firstInSample = false;
    }

    if (verbose)
    {
        uint64_t hostDiff = hosttime - previousWriteHostTime;
        previousWriteHostTime = hosttime;
        uint64_t sampleDiff = ((inSamples.TimeStamp - previousWriteSampleTime) * NANOSECOND) / format.SampleRate;
        previousWriteSampleTime = inSamples.TimeStamp;

        LOGD_VERBOSE((dipo, "%s audio in: host %" PRIu64 ", sample %" PRIu64 ", host diff %" PRIu64 ", sample diff %" PRIu64,
                (channel == AudioChannelType_Main ? "main" : "alt"), hosttime, inSamples.TimeStamp,
                hostDiff, sampleDiff));
    }

    // hostTime is currently not used by Apple communications plug-in
    uint64_t hostTime = hostInStartTime + (inSamples.TimeStamp * NANOSECOND / format.SampleRate);

    #if 0
        // Apple communications plug-in currently expects sample number for inSampleTime,
        // and not the timestamp
        inputCallback(inSamples.TimeStamp, hostTime, inSamples.DataPtr, inSamples.Length, inputContext);
    #else
        inputCallback(sampleNumber, hostTime, inSamples.DataPtr, inSamples.Length, inputContext);
    #endif

    sampleNumber += inSamples.Length / (format.Channels * format.BitsPerChannel / 8);
}

} } // namespace adit { namespace carplay
